"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
var _devServer = require("@expo/dev-server");
var _assert = _interopRequireDefault(require("assert"));
var _chalk = _interopRequireDefault(require("chalk"));
var Log = _interopRequireWildcard(require("../../log"));
var _link = require("../../utils/link");
var _prompts = require("../../utils/prompts");
var _commandsTable = require("./commandsTable");
function _interopRequireDefault(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
function _interopRequireWildcard(obj) {
    if (obj && obj.__esModule) {
        return obj;
    } else {
        var newObj = {};
        if (obj != null) {
            for(var key in obj){
                if (Object.prototype.hasOwnProperty.call(obj, key)) {
                    var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {};
                    if (desc.get || desc.set) {
                        Object.defineProperty(newObj, key, desc);
                    } else {
                        newObj[key] = obj[key];
                    }
                }
            }
        }
        newObj.default = obj;
        return newObj;
    }
}
const debug = require("debug")("expo:start:interface:interactiveActions");
class DevServerManagerActions {
    constructor(devServerManager){
        this.devServerManager = devServerManager;
    }
    printDevServerInfo(options) {
        // If native dev server is running, print its URL.
        if (this.devServerManager.getNativeDevServerPort()) {
            const devServer = this.devServerManager.getDefaultDevServer();
            try {
                const nativeRuntimeUrl = devServer.getNativeRuntimeUrl();
                const interstitialPageUrl = devServer.getRedirectUrl();
                (0, _commandsTable).printQRCode(interstitialPageUrl != null ? interstitialPageUrl : nativeRuntimeUrl);
                if (interstitialPageUrl) {
                    Log.log((0, _commandsTable).printItem(_chalk.default`Choose an app to open your project at {underline ${interstitialPageUrl}}`));
                }
                Log.log((0, _commandsTable).printItem(_chalk.default`Metro waiting on {underline ${nativeRuntimeUrl}}`));
                // TODO: if development build, change this message!
                Log.log((0, _commandsTable).printItem("Scan the QR code above with Expo Go (Android) or the Camera app (iOS)"));
            } catch (error) {
                // @ts-ignore: If there is no development build scheme, then skip the QR code.
                if (error.code !== "NO_DEV_CLIENT_SCHEME") {
                    throw error;
                } else {
                    const serverUrl = devServer.getDevServerUrl();
                    Log.log((0, _commandsTable).printItem(_chalk.default`Metro waiting on {underline ${serverUrl}}`));
                    Log.log((0, _commandsTable).printItem(`Linking is disabled because the client scheme cannot be resolved.`));
                }
            }
        }
        const webDevServer = this.devServerManager.getWebDevServer();
        const webUrl = webDevServer == null ? void 0 : webDevServer.getDevServerUrl({
            hostType: "localhost"
        });
        if (webUrl) {
            Log.log();
            Log.log((0, _commandsTable).printItem(_chalk.default`Web is waiting on {underline ${webUrl}}`));
        }
        (0, _commandsTable).printUsage(options, {
            verbose: false
        });
        (0, _commandsTable).printHelp();
        Log.log();
    }
    async openJsInspectorAsync() {
        Log.log("Opening JavaScript inspector in the browser...");
        const metroServerOrigin = this.devServerManager.getDefaultDevServer().getJsInspectorBaseUrl();
        (0, _assert).default(metroServerOrigin, "Metro dev server is not running");
        const apps = await (0, _devServer).queryAllInspectorAppsAsync(metroServerOrigin);
        if (!apps.length) {
            Log.warn(`No compatible apps connected. JavaScript Debugging can only be used with the Hermes engine. ${(0, _link).learnMore("https://docs.expo.dev/guides/using-hermes/")}`);
            return;
        }
        try {
            for (const app of apps){
                await (0, _devServer).openJsInspector(app);
            }
        } catch (error) {
            Log.error("Failed to open JavaScript inspector. This is often an issue with Google Chrome.");
            Log.exception(error);
        }
    }
    reloadApp() {
        Log.log(`${_commandsTable.BLT} Reloading apps`);
        // Send reload requests over the dev servers
        this.devServerManager.broadcastMessage("reload");
    }
    async openMoreToolsAsync() {
        try {
            // Options match: Chrome > View > Developer
            const value = await (0, _prompts).selectAsync(_chalk.default`Dev tools {dim (native only)}`, [
                {
                    title: "Inspect elements",
                    value: "toggleElementInspector"
                },
                {
                    title: "Toggle performance monitor",
                    value: "togglePerformanceMonitor"
                },
                {
                    title: "Toggle developer menu",
                    value: "toggleDevMenu"
                },
                {
                    title: "Reload app",
                    value: "reload"
                }, 
            ]);
            this.devServerManager.broadcastMessage("sendDevCommand", {
                name: value
            });
        } catch (error) {
            debug(error);
        // do nothing
        } finally{
            (0, _commandsTable).printHelp();
        }
    }
    toggleDevMenu() {
        Log.log(`${_commandsTable.BLT} Toggling dev menu`);
        this.devServerManager.broadcastMessage("devMenu");
    }
}
exports.DevServerManagerActions = DevServerManagerActions;

//# sourceMappingURL=interactiveActions.js.map