"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
exports.validateBundleId = validateBundleId;
exports.validatePackage = validatePackage;
exports.assertValidBundleId = assertValidBundleId;
exports.assertValidPackage = assertValidPackage;
exports.getBundleIdWarningAsync = getBundleIdWarningAsync;
exports.getPackageNameWarningAsync = getPackageNameWarningAsync;
var _assert = _interopRequireDefault(require("assert"));
var _chalk = _interopRequireDefault(require("chalk"));
var _client = require("../api/rest/client");
var _link = require("./link");
var _url = require("./url");
function _interopRequireDefault(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const debug = require("debug")("expo:utils:validateApplicationId");
const IOS_BUNDLE_ID_REGEX = /^[a-zA-Z0-9-.]+$/;
const ANDROID_PACKAGE_REGEX = /^[a-zA-Z][a-zA-Z0-9_]*(\.[a-zA-Z][a-zA-Z0-9_]*)+$/;
function validateBundleId(value) {
    return IOS_BUNDLE_ID_REGEX.test(value);
}
function validatePackage(value) {
    return ANDROID_PACKAGE_REGEX.test(value);
}
function assertValidBundleId(value) {
    _assert.default.match(value, IOS_BUNDLE_ID_REGEX, `The ios.bundleIdentifier defined in your Expo config is not formatted properly. Only alphanumeric characters, '.', '-', and '_' are allowed, and each '.' must be followed by a letter.`);
}
function assertValidPackage(value) {
    _assert.default.match(value, ANDROID_PACKAGE_REGEX, `Invalid format of Android package name. Only alphanumeric characters, '.' and '_' are allowed, and each '.' must be followed by a letter.`);
}
const cachedBundleIdResults = {};
const cachedPackageNameResults = {};
async function getBundleIdWarningAsync(bundleId) {
    // Prevent fetching for the same ID multiple times.
    if (cachedBundleIdResults[bundleId]) {
        return cachedBundleIdResults[bundleId];
    }
    if (!await (0, _url).isUrlAvailableAsync("itunes.apple.com")) {
        debug(`Couldn't connect to iTunes Store to check bundle ID ${bundleId}. itunes.apple.com may be down.`);
        // If no network, simply skip the warnings since they'll just lead to more confusion.
        return null;
    }
    const url = `http://itunes.apple.com/lookup?bundleId=${bundleId}`;
    try {
        debug(`Checking iOS bundle ID '${bundleId}' at: ${url}`);
        const response = await (0, _client).fetchAsync(url);
        const json = await response.json();
        if (json.resultCount > 0) {
            const firstApp = json.results[0];
            const message = formatInUseWarning(firstApp.trackName, firstApp.sellerName, bundleId);
            cachedBundleIdResults[bundleId] = message;
            return message;
        }
    } catch (error) {
        debug(`Error checking bundle ID ${bundleId}: ${error.message}`);
    // Error fetching itunes data.
    }
    return null;
}
async function getPackageNameWarningAsync(packageName) {
    // Prevent fetching for the same ID multiple times.
    if (cachedPackageNameResults[packageName]) {
        return cachedPackageNameResults[packageName];
    }
    if (!await (0, _url).isUrlAvailableAsync("play.google.com")) {
        debug(`Couldn't connect to Play Store to check package name ${packageName}. play.google.com may be down.`);
        // If no network, simply skip the warnings since they'll just lead to more confusion.
        return null;
    }
    const url = `https://play.google.com/store/apps/details?id=${packageName}`;
    try {
        debug(`Checking Android package name '${packageName}' at: ${url}`);
        const response = await (0, _client).fetchAsync(url);
        // If the page exists, then warn the user.
        if (response.status === 200) {
            // There is no JSON API for the Play Store so we can't concisely
            // locate the app name and developer to match the iOS warning.
            const message = `⚠️  The package ${_chalk.default.bold(packageName)} is already in use. ${_chalk.default.dim((0, _link).learnMore(url))}`;
            cachedPackageNameResults[packageName] = message;
            return message;
        }
    } catch (error) {
        debug(`Error checking package name ${packageName}: ${error.message}`);
    // Error fetching play store data or the page doesn't exist.
    }
    return null;
}
function formatInUseWarning(appName, author, id) {
    return `⚠️  The app ${_chalk.default.bold(appName)} by ${_chalk.default.italic(author)} is already using ${_chalk.default.bold(id)}`;
}

//# sourceMappingURL=validateApplicationId.js.map