#version 410

//#define MAX_LIGHTS 3

#pragma include "p3d/uniform.glsl"
#pragma include "lib/shadow.glsl"
#pragma include "lib/constants.glsl"
#pragma include "lib/color.glsl"
#pragma include "lib/math.glsl"

// Uniform inputs
uniform float u_MaxShininess;
uniform float u_RimLightIntensity;
uniform float u_ShadowSamples;
uniform float u_ShadowSpread;
uniform float u_SpecularBlendStrength;
uniform int u_NumberOfLights;
uniform float u_SpecularPower;
uniform float u_Roughness;
uniform float u_Brightness;
uniform float u_Saturation;

uniform float u_Reflectivity;

// Input from vertex shader
in vec4 vertInShadowSpaces[MAX_LIGHTS];
in vec4 vertColor;
in vec4 vertPosition;
in vec2 vertMultiTexCoord0;
in vec3 vertNormal;
in vec3 vertObjPosition;

// Output to the screen
layout(location = 0) out vec4 fragColor;
layout(location = 1) out vec4 reflectivityOut;

#define MAX_FRESNEL_POWER 5.0
#define MAX_SHININESS     127.75

// Clamp value to [0,1] (short alias)
float sat(float x) { return clamp(x, 0.0, 1.0); }

float hash3(vec3 p) {
    p = fract(p * 0.1031);
    p += dot(p, p.yzx + 33.33);
    return fract((p.x + p.y) * p.z);
}

void main() {

    vec4 diffuseColor = vertColor;

    if (diffuseColor.a < hash3(vertObjPosition)) discard;

    diffuseColor = color_ops(diffuseColor, u_Saturation, u_Brightness);

    float shininess = MAX_SHININESS * u_MaxShininess;

    vec4 diffuse  = vec4(0.0, 0.0, 0.0, diffuseColor.a);
    vec4 specular = vec4(0.0, 0.0, 0.0, diffuseColor.a);
    vec4 rim      = vec4(0.0, 0.0, 0.0, diffuseColor.a);

    vec3 normal = normalize(vertNormal);

    vec3 eyeDirection = normalize(-vertPosition.xyz);

    for (int i = 0; i < u_NumberOfLights; ++i) {  // i = 0

        vec3 lightDirection = p3d_LightSource[i].position.xyz - vertPosition.xyz * p3d_LightSource[i].position.w;

        vec3 unitLightDirection = normalize(lightDirection);
        vec3 halfwayDirection   = normalize(unitLightDirection + eyeDirection);

        float lightDistance = length(lightDirection);

        // Diffuse
        float diffuseIntensity  = dot(normal, unitLightDirection);
        if (diffuseIntensity < 0.0) { continue; }
        vec4 diffuseTemp = vec4(
            clamp(diffuseColor.rgb * p3d_LightSource[i].diffuse.rgb * diffuseIntensity, 0.0 , 1.0), diffuseColor.a
        );

        // Specular
        float specularIntensity = sat(dot(normal, halfwayDirection));
        vec4 lightSpecularColor = p3d_LightSource[i].specular;
        vec4 materialSpecularColor = vec4(vec3(1.0), diffuseColor.a);

        float f = 1.0 - max(dot(halfwayDirection, eyeDirection), 0.0);
        float fresnelFactor = f * f * f; // zamiast pow(f, 5.0)

        vec4 specularTemp = vec4(vec3(0.0), diffuseColor.a);
        specularTemp.rgb  = lightSpecularColor.rgb * pow(specularIntensity, shininess);
        specularTemp.rgb *= materialSpecularColor.rgb;
        specularTemp.rgb  = clamp(specularTemp.rgb, 0.0, 1.0);  // Comment to have more specular
        specularTemp.rgb *= u_SpecularPower;

        // Spot
        float unitLightDirectionDelta = dot(normalize(p3d_LightSource[i].spotDirection), -unitLightDirection);
        if (unitLightDirectionDelta < p3d_LightSource[i].spotCosCutoff) { continue; }
        float spotExponent = p3d_LightSource[i].spotExponent;
        diffuseTemp.rgb *= (spotExponent <= 0.0 ? 1.0 : pow(unitLightDirectionDelta, spotExponent));

        // Shadow
        float shadow = sh(
            p3d_LightSource[i].shadowMap,
            vertInShadowSpaces[i],
            u_ShadowSpread,
            u_ShadowSamples // float selector: 0..4
        );

        diffuseTemp.rgb  *= shadow;
        specularTemp.rgb *= shadow;

        diffuse  += diffuseTemp;
        specular += specularTemp;

        // Rim
        vec4 rimLight = vec4(0.0);
        float rimFactor = 1.0 - max(0.0, dot(eyeDirection, normal));
        rimLight.rgb = vec3(rimFactor);
        rimLight.rgb *= rimLight.rgb;         // x^2
        rimLight.rgb *= u_RimLightIntensity;
        rimLight.rgb *= diffuse.rgb;
        rim += rimLight;

    }

    vec4 ambient = p3d_Material.ambient * p3d_LightModel.ambient * diffuseColor;

    vec3 color = diffuse.rgb + rim.rgb + specular.rgb + ambient.rgb + p3d_Material.emission.rgb;

    fragColor = vec4(color, diffuseColor.a);
    reflectivityOut = vec4(vec3(u_Reflectivity), 1.0);

}
