unit comAsync;          { Asynchronous Communications Object}

{ Code partially based upon Mike Fricker's OOP Communications Device Unit }

{
     comAsynch - Copyright 1998 by Brian Zhou
     You are free to use this code in your programs, however
     it may not be included in Source/TPU function libraries
     without my permission.

     You may reach me at zoob@darktech.org
}

interface

uses comAbs;

var
  oport : word;
  pirq  : byte;
  pbase : word;

type
  parity = (async_none, async_even, async_odd, async_zero, async_one);

  oAsyncComm = object(oAbsComm)
    constructor init;
    function    install(cport, baud : word) : boolean; virtual;
    procedure   deinstall;                      virtual;
    procedure   send(c : char);                 virtual;
    function    receive : char;                 virtual;
    function    waiting : boolean;              virtual;
    function    carrier : boolean;              virtual;
    procedure   fRx;                            virtual;
    procedure   fTx;                            virtual;
    procedure   pRx;                            virtual;
    procedure   pTx;                            virtual;
    procedure   dtr(state : boolean);           virtual;
    destructor  done;
  end;
  pAsyncComm = ^oAsyncComm;

implementation

Uses dos;

Type
   tBuffer = array[0..4096] of Char;

Var
   Active         : Boolean;
   R_Buffer       : ^tBuffer;
   R_Head         : Word;
   R_Tail         : Word;
   R_Size         : Word;
   T_Buffer       : ^tBuffer;
   T_Head         : Word;
   T_Tail         : Word;
   T_Size         : Word;
   UART_Data      : Word;
   UART_IER       : Word;
   UART_IIR       : Word;
   UART_LCR       : Word;
   UART_MCR       : Word;
   UART_LSR       : Word;
   UART_MSR       : Word;
   OLD_MCR        : Byte;
   Org_Vector     : Pointer;
   commTsize, commRsize : Word;


const
   portBase : array[1..8] of Word = ($03F8,$02F8,$03E8,$02E8,
                                     $107C,$1084,$1464,$146C);
   portIRQ  : array[1..8] of Byte = (    4,    3,    3,    3,
                                         3,    3,    3,    3);

{ %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%[ oAsyncCom methods ]%%% }

{ ::: enable interrupts }
procedure enableIntr;
begin
  inline($fa); {cli}
end;

{ ::: disable interrupts }
procedure disableIntr;
begin
  inline($fb); {sti}
end;

{ ::: direct interrupt driver }
procedure commInterrupt; interrupt;
const IIR : Byte = 0;
  begin
    if Active Then Begin
      iir := Port[UART_IIR];
      while not Odd(IIR) do begin
        case (iir shr 1) of
          0 : iir := Port[UART_MSR];
          1 : if T_Head = T_Tail then Port[UART_IER] := Port[UART_IER] and not 2 else
              begin
                Port[UART_DATA] := Byte(T_Buffer^[T_Head]);
                Inc(T_Head);
                if T_Head > T_Size then T_Head := 0;
              end;
          2 : begin
                R_Buffer^[R_Tail] := Char(Port[Uart_Data]);
                Inc(R_Tail);
                if R_Tail > R_Size then R_Tail := 0;
                if R_Tail = R_Head then begin
                  Inc(R_Head);
                  if R_Head > R_Size then R_Head := 0;
                end;
              end;
          3 : iir := Port[UART_LSR];
        end;
        iir := Port[UART_IIR];
      end;
    end;
    Port[$20] := $20;
  end;

{ ::: construct object }
constructor oAsyncComm.init;
  begin
    Active    := False;
    commTsize := 4096; commRsize := 4096;
    R_Buffer  := nil;  R_Head    := 0;  R_Tail    := 0;  R_Size    := 0;
    T_Buffer  := nil;  T_Head    := 0;  T_Tail    := 0;  T_Size    := 0;
    pBase     := 0;    pIrq      := 0;
  end;

{ ::: driver installer }
function oAsyncComm.install(cport,baud : word) : boolean;
var inUse : Boolean; Ktr, lcr : Byte; divs:word;
  begin
     install := False;  divs := 115200 div baud;  oPort := cPort;

     if (cport < 1) or (active) then Exit;

     R_Size := 255;  T_Size := 255;

     if pBase = 0 then pBase := portBase[oPort];
     if pIrq  = 0 then pIrq  := portIrq [oPort];

     GetMem(R_Buffer,commRsize);    R_Size := commRsize;
     GetMem(T_Buffer,commTsize);    T_Size := commTsize;

     UART_DATA := pBase+0;  UART_IER  := pBase+1;  UART_IIR  := pBase+2;
     UART_LCR  := pBase+3;  UART_MCR  := pBase+4;  UART_LSR  := pBase+5;
     UART_MSR  := pBase+6;

     enableIntr;

     Port[$21] := Port[$21] or (1 shl pIrq);
     GetIntVec(8+pIrq,Org_Vector);
     SetIntVec(8+pIrq,@commInterrupt);
     Port[$21] := Port[$21] and not (1 shl pIrq);

     Old_MCR := Port[UART_MCR];
     Port[UART_LCR] := 3;  Port[UART_IER] := 1;

     Active := True;

     Port [uart_lcr ] := Port[uart_lcr] or $80;
     Portw[uart_Data] := divs;
     Port [uart_lcr]  := Port[uart_lcr] and not $80;
     lcr:= $00 or $03;

     Port[Uart_lcr] := Port[uart_lcr] and $40 or lcr;
     Port[Uart_MCR] := 11;
     disableIntr;

     install := True;
  end;

{ ::: driver deinstaller }
procedure oAsyncComm.deInstall;
  begin
    enableIntr;

    Port[UART_MCR] := Old_MCR;
    Port[UART_IER] := 0;

    Port[$21] := Port[$21] or ($01 shr pIrq);
    SetIntVec(8+pIrq,Org_Vector);

    disableIntr;

    Freemem(R_Buffer,R_Size);  Freemem(T_Buffer,T_Size);
    Active := False; oPort := 0;
  end;

{ ::: transmit character }
procedure oAsyncComm.send(c : char);
  begin
    T_Buffer^[T_Tail] := c;
    Inc(T_Tail);
    if T_Tail > T_Size then T_Tail := 0;
    if T_Tail = T_Head then begin
      Inc(T_Head);
      if T_Head > T_Size then T_Head := 0;
    end;
    enableIntr;
    Port[UART_IER] := Port[UART_IER] or 2;
    disableIntr;
  end;

{ ::: receive character }
function oAsyncComm.receive : char;
  begin
    receive := #0;
    if R_Head = R_Tail then receive := #0 else begin
      receive := R_Buffer^[R_Head];
      Inc(R_Head);
      if R_Head > R_Size then R_Head := 0;
    end;
  end;

{ ::: characters waiting to be received? }
function oAsyncComm.waiting : boolean;
  begin
    Waiting := R_Head <> R_Tail;
  end;

{ ::: carrier present? }
function oAsyncComm.carrier : boolean;
  begin
    carrier := Port[UART_MSR] and $80 > 0;
  end;

{ ::: flush receive buffer }
procedure oAsyncComm.fRx;
  begin
    R_Tail := 0;
    R_Head := 0;
  end;

{ ::: flush transmit buffer }
procedure oAsyncComm.fTx;
  begin
    T_Tail := 0;
    T_Head := 0;
  end;

{ ::: purge receive buffer }
procedure oAsyncComm.pRx;
  begin
    oAsyncComm.fRx;
  end;

{ ::: purge transmit buffer }
procedure oAsyncComm.pTx;
  begin   
    oAsyncComm.fTx;
  end;

{ ::: toggle dtr }
procedure oAsyncComm.dtr(state : boolean);
  begin
    if state then
      port[uart_mcr]:=port[uart_mcr] or 1
    else
      port[uart_mcr]:=port[uart_mcr] and not 1;
  end;

{ ::: destruct object }
destructor oAsyncComm.done;
  begin
  end;

end.
